-- PG ErrorCode: 'C0001' = SecPol Verstoss (https://www.postgresql.org/docs/current/errcodes-appendix.html)

-- 34060 'SecPol Verstoß'
-- 34061 'SecPol Verstoß für Tabelle %s'
-- 34062 'Rollen-Mitgliedschaft für aktuellen Benutzer überprüfen oder SecPol Prozeduren/Trigger anpassen'
-- 34063 'Fehlende SecPol Implmentierung für '
-- 34064 'Trigger für SecPol an Tabelle angehängt, jedoch keine Prozedur definiert'
-- 34065 'Entweder Prozedur definieren oder Trigger entfernen (unnötiger Code wird ausgeführt, wenn keine Prozedur(Check) definiert ist)'
--
-- 34066 Customer: SecPol Check Error Hint (art)
-- 34067 Customer: SecPol Check Error Hint (ldsdok)
-- 34068 Customer: SecPol Check Error Hint (recnocomments)
-- 34069 Customer: SecPol Check Error Hint (opl)
-- 34070 Customer: SecPol Check Error Hint (op2)
-- 34071 Customer: SecPol Check Error Hint (op6)
-- 34072 Customer: SecPol Check Error Hint (stv)
-- 34073 Customer: SecPol Check Error Hint (epreis)
-- 34074 Customer: SecPol Check Error Hint (ab2)
-- 34075 Customer: SecPol Check Error Hint (auftg)

-- ╔══════════════════════════════════════════════════════════════════════════════╗
-- ‖ Zentrale Trigger Funktion: TSystem.alltables__b_iud__secpol_check()          ‖
-- ‖     CIMPCS checks                                                            ‖
-- ╟━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━╢
-- ‖                                                                              ‖
-- ‖ CREATE TRIGGER <TABLE>__b_999_iud__secpol_check                              ‖
-- ‖ BEFORE INSERT OR UPDATE OR DELETE                                            ‖
-- ‖ ON <TABLE>                                                                   ‖
-- ‖ FOR EACH ROW                                                                 ‖
-- ‖ EXECUTE FUNCTION TSystem.alltables__b_iud__secpol_check();                   ‖
-- ╚══════════════════════════════════════════════════════════════════════════════╝
--
-- ╔══════════════════════════════════════════════════════════════════════════════╗
-- ‖ Zentrale Trigger Funktion: TSystem.alltables__b_iud__secpol_customer_check() ‖
-- ‖ Customer specific checks                                                     ‖
-- ╟━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━╢
-- ‖                                                                              ‖
-- ‖ CREATE TRIGGER <TABLE>__b_999_iud__secpol_customer_check                     ‖
-- ‖ BEFORE INSERT OR UPDATE OR DELETE                                            ‖
-- ‖ ON <TABLE>                                                                   ‖
-- ‖ FOR EACH ROW                                                                 ‖
-- ‖ EXECUTE FUNCTION TSystem.alltables__b_iud__secpol_customer_check();                   ‖
-- ╚══════════════════════════════════════════════════════════════════════════════╝
--                             ⇓⇓⇓
-- ╔══════════════════╤════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════╗
-- ‖ INSERT / DELETE  │ UPDATE                                                                                                                 ‖
-- ╟──────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────╢
-- ‖                  │ _ignore := TSystem.secpol_compare_records_oldnew(_rec_old, _rec_current, _cols_check, _cols_ignore);  -- sind gleich   ‖
-- ‖        ⇓         │ IF (NOT(_ignore)) THEN                                                                                                 ‖
-- ‖                  │     ⇓                                                                                                                  ‖
-- ╟──────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────╢
-- ‖ !!! SEARCHPATH erweitert um z_50_customer für alltables__b_iud__secpol_customer_check !!!                                                 ‖
-- ‖ !!! Sollte z_50_customer keine secpol_customer_check Funktion für <TABLE> bereitstellen,                                                  ‖
-- ‖     lösst die Default-Implementierung eine Exception aus                                                                                  ‖
-- ‖                                                                                                                                           ‖
-- ‖ SELECT                                                                                                                                    ‖
-- ‖   (secpol_check(<TABLE>, recold_as_json, recnew_as_jsonnew)).*                                                                            ‖
-- ‖ INTO                                                                                                                                      ‖
-- ‖   _ok, _err_hint;                                                                                                                         ‖
-- ╟───────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────╢
-- ‖ PERFORM TSystem.secpol_raise_exception(NOT(_ok), NULL, NULL, _err_hint::TEXT, 'C0001'::TEXT, NULL, NULL, tg_relname, NULL);               ‖
-- ╚═══════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════╝
--
-- ╔══════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════╗
-- ‖ Definiere secpol_check für <TABLE> in schema tsystem (file: \0300 Functions\999 secpol-check.sql)                                                                            ‖
-- ‖ Definiere secpol_customer_check für <TABLE> in schema z_50_customer (file: \9100-customer-functions\z99-<customer>\secpol-<customer>)                                        ‖
-- ╟━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━╢
-- ‖                                                                                                                                                                              ‖
-- ‖ CREATE OR REPLACE FUNCTION tsystem.secpol_check(IN row_data art, IN rec_old JSONB, IN rec_new JSONB, OUT check_ok BOOL, OUT hint VARCHAR) RETURNS RECORD AS                  ‖
-- ‖ CREATE OR REPLACE FUNCTION z_50_customer.secpol_customer_check(IN row_data art, IN rec_old JSONB, IN rec_new JSONB, OUT check_ok BOOL, OUT hint VARCHAR) RETURNS RECORD AS   ‖
-- ‖ $$                                                                                                                                                                           ‖
-- ‖ BEGIN                                                                                                                                                                        ‖
-- ‖   hint := prodat_languages.lang_text(<SIEHE OBEN DFINIERTE TEXT0-ID>)::VARCHAR; -- placeholder id, customer needs to actually set a meaningfull text for that id             ‖
-- ‖                                                                                                                                                                              ‖
-- ‖   -- INSERT UPDATE DELETE                                                                                                                                                    ‖
-- ‖   check_ok := < liefere Ergebniss basierend auf rec_old, rec_new und current_user/Berechtigung >;                                                                            ‖
-- ‖ END                                                                                                                                                                          ‖
-- ‖ $$ LANGUAGE plpgsql STABLE;                                                                                                                                                  ‖
-- ‖                                                                                                                                                                              ‖
-- ╟──────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────╢
-- ‖  Kann benutzt werden, um aus einer Child-Table heraus den Check an die Parent-table zu delegieren                                                                            ‖
-- ‖  TSystem.secpol_check_exec_parent(                                                                                                                                           ‖
-- ‖    IN tablename VARCHAR,                                                                                                                                                     ‖
-- ‖    IN dbrid_old VARCHAR,                                                                                                                                                     ‖
-- ‖    IN dbrid_new VARCHAR,                                                                                                                                                     ‖
-- ‖    OUT check_ok BOOL,                                                                                                                                                        ‖
-- ‖    OUT hint VARCHAR                                                                                                                                                          ‖
-- ‖  )                                                                                                                                                                           ‖
-- ╚══════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════════╝





-- function to raise exception wich allows to pass in parameters for all the parts of an pg exception
-- with predefined defaults for the use case of secpol
CREATE OR REPLACE FUNCTION TSystem.secpol_raise_exception(
    err_doraise    BOOL DEFAULT false,
    err_message    TEXT DEFAULT NULL,
    err_detail     TEXT DEFAULT NULL,
    err_hint       TEXT DEFAULT NULL,
    err_code       TEXT DEFAULT NULL,
    err_constraint TEXT DEFAULT NULL,
    err_schema     TEXT DEFAULT NULL,
    err_table      TEXT DEFAULT NULL,
    err_column     TEXT DEFAULT NULL
  )
    RETURNS BOOL
    immutable
    parallel safe
    LANGUAGE plpgsql
    SET search_path = 'public'
    AS
  $$
  DECLARE
    _Detail VARCHAR;

  BEGIN
    _Detail = FORMAT(coalesce(err_detail, prodat_languages.lang_text(34061)), coalesce(err_table, '?'));
    IF (err_doraise) THEN
      RAISE EXCEPTION USING
        message     = 'secpol_exception ' || coalesce(err_message, 'xtt34060'),
        detail      = _Detail,
        hint        = coalesce(err_hint, prodat_languages.lang_text(34062)),
        errcode     = coalesce(err_code, 'raise_exception'), -- ERRCODE_RAISE_EXCEPTION (P0001) / DO NOT TRANSLATE
        constraint  = coalesce(err_constraint, ''),
        schema      = coalesce(err_schema, ''),
        table       = coalesce(err_table, ''),
        column      = coalesce(err_column, '')
        --datatype    = pg_typeof(value)::TEXT;
      ;
    END IF;

    RETURN TRUE;
  END;
  $$;

-- allows comparing 2 JSONB objects (created by converting a tablerow to JSONB, thus comparing 2 rows)
-- by restricting which columns are actually compared
-- returns true if the 2 are the same (having the compared columns with the same value)
CREATE OR REPLACE FUNCTION TSystem.secpol_compare_records_oldnew(
    rec_old      JSONB,
    rec_new      JSONB,
    cols_checked VARCHAR[],                        -- if empty is the same as if all columns would be included
    cols_ignored VARCHAR[] DEFAULT '{}'::VARCHAR[] -- every column from this array will be ignored, no matter if its included in cols_checked
  )
    RETURNS BOOL
    immutable
    parallel safe
    LANGUAGE plpgsql
    AS
  $$
  DECLARE
    _old JSONB;
    _new JSONB;
    _cols_ignored VARCHAR[];

  BEGIN
    _cols_ignored := cols_ignored || '{dbrid,insert_date,insert_by,modified_date,modified_by}'::VARCHAR[];

    SELECT
      jsonb_object_agg(json_pairs.key, json_pairs.value)
    FROM
      jsonb_each(rec_old) AS json_pairs
    WHERE
      ( (cardinality(cols_checked) = 0) OR (key = ANY(cols_checked)) )
      AND NOT(key = ANY(_cols_ignored))
    INTO
      _old
    ;

    SELECT
      jsonb_object_agg(json_pairs.key, json_pairs.value)
    FROM
      jsonb_each(rec_new) AS json_pairs
    WHERE
      ( (cardinality(cols_checked) = 0) OR (key = ANY(cols_checked)) )
      AND NOT(key = ANY(_cols_ignored))
    INTO
      _new
    ;

    RETURN ( (_old @> _new) AND (_new @> _old) );  -- NOT changed;
  END;
  $$;

-- the default secpol_check procedure, which is called when no procedure for that particular table is defined
-- raises exception, because having a trigger for secpol defined but no secpol_check(<table>, ...) function: error
CREATE OR REPLACE FUNCTION TSystem.secpol_check(IN row_data ANYELEMENT, IN rec_old JSONB, IN rec_new JSONB, OUT check_ok BOOL, OUT hint VARCHAR) RETURNS RECORD AS
  $$
  BEGIN
    PERFORM TSystem.secpol_raise_exception
    (
      true,
      CONCAT(prodat_languages.lang_text(34063), pg_typeof(row_data)),
      prodat_languages.lang_text(34064),
      prodat_languages.lang_text(34065),
      'C0001'::TEXT,
      NULL,
      NULL,
      pg_typeof(row_data)::TEXT,
      NULL
    );
  END
  $$ LANGUAGE plpgsql;

-- the default secpol_customer_check procedure, which is called when no procedure for that particular table is defined
-- raises exception, because having a trigger for secpol(customer) defined but no secpol_customer_check(<table>, ...) function: error
CREATE OR REPLACE FUNCTION TSystem.secpol_customer_check(IN row_data ANYELEMENT, IN rec_old JSONB, IN rec_new JSONB, OUT check_ok BOOL, OUT hint VARCHAR) RETURNS RECORD AS
  $$
  BEGIN
    PERFORM TSystem.secpol_raise_exception
    (
      true,
      CONCAT(prodat_languages.lang_text(34063), pg_typeof(row_data)),
      prodat_languages.lang_text(34064),
      prodat_languages.lang_text(34065),
      'C0001'::TEXT,
      NULL,
      NULL,
      pg_typeof(row_data)::TEXT,
      NULL
    );
  END
  $$ LANGUAGE plpgsql;

-- executes the appropiate z_50_customer.secpol_check function for the given table by its name
-- the only place where 'magic string's are used for secpol_check and %I is used (identifiers)
--DROP FUNCTION TSystem.secpol_check_exec_parent(VARCHAR, VARCHAR, VARCHAR);
CREATE OR REPLACE FUNCTION TSystem.secpol_check_exec_parent(
    IN tablename VARCHAR,
    IN dbrid_old VARCHAR,
    IN dbrid_new VARCHAR,
    OUT check_ok BOOL,
    OUT hint VARCHAR
  )
    RETURNS RECORD
    SET search_path = public, TSystem, z_50_customer
    AS
  $$
  DECLARE
    _ret BOOL;
    _sql VARCHAR;

  BEGIN
    -- execute the check function for the parent record / correct function is selected by parameter type when executing
    _sql :=
      FORMAT
      (
        $SQL$
          WITH
            _input AS
            (
              SELECT
                (
                  coalesce((SELECT to_jsonb( %1$I ) FROM %1$I WHERE dbrid = $1), '{}')
                ) AS rec_old,
                (
                  coalesce((SELECT to_jsonb( %1$I ) FROM %1$I WHERE dbrid = $2), '{}')
                ) AS rec_new
            )
          SELECT
            (secpol_check(null::%1$I, rec_old, rec_new)).*
          FROM
            _input
          ;
        $SQL$,
        tablename
      )
    ;

    EXECUTE
      _sql
    INTO
      check_ok,
      hint
    USING
      dbrid_old, dbrid_new
    ;
  END
  $$ LANGUAGE plpgsql;

-- retrieve the array of columns to be checked and ignored during secpol_check when an update of a record happens
-- to determine the update performed is to be ignored for the secpol_check
-- # system columns like dbrid, modified_by, modified_date etcpp are ignored by default in secpol_compare_records_oldnew()
-- # record might be implicitely updated by triggers on columns which are not directly edited by users
CREATE OR REPLACE FUNCTION TSystem.secpol_checkcolumns_get(IN row_data anyelement, OUT cols_check VARCHAR[], OUT cols_ignore VARCHAR[]) RETURNS RECORD
  AS $$
  DECLARE
    table_type constant regtype := pg_typeof(row_data);

  BEGIN
    -- as function, so at a later time the implementation can be changed (remove IMMUTABLE when doing so !!!)
    -- eg. to retrieve the columns from a table

    CASE table_type
      -- art -- Spalten definieren
      WHEN pg_typeof(null::art) THEN
        cols_check :=
          (
            '{'
              'ak_copy_exact,'
              -- Hauptfelder
              'ak_fertigung, ak_nr, ak_bez, ak_mat, ak_dim, ak_znr, ak_idx, ak_standard_mgc, ak_hersteller, ak_herstelleraknr,'
              -- QS
              'ak_din, ak_neuanlage, ak_gefahrgut, ak_norm, ak_ersatzaknr,'
              -- Lager
              'ak_sernrreq, ak_chnrreq, ak_konsre'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- ldsdok -- Spalten definieren => UNUSED?!
      WHEN pg_typeof(null::ldsdok) THEN
        cols_check :=
          (
            '{'
              -- Hauptfelder
              'ld_aknr, ld_aknr_idx, ld_akbz,'
              -- Adressfelder
              'ld_kn,'
              -- Texte
              -- Zuschnittsangaben
              'ld_o6_dimi, ld_o6_stkz, ld_o6_lz, ld_o6_bz, ld_o6_hz, ld_o6_zme'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];
      -- opl -- Spalten definieren
      WHEN pg_typeof(null::opl) THEN
        cols_check :=
          (
            '{'
              'op_n, op_ix, '
              'op_vi, op_vit, op_stat, '
              'op_txt, op_txt_rtf, '
              'op_quali, op_quali_rtf, '
              'op_pruef, op_pruef_rtf'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- op2 -- Spalten definieren
      WHEN pg_typeof(null::op2) THEN
        cols_check :=
          (
            '{'
              'o2_ix, o2_n, o2_ks, '
              'o2_txt, o2_txt_rtf, o2_rustinfo, '
              'o2_aw, o2_aknr, o2_awtx, o2_adkrz, '
              'o2_fert_prozess, o2_msp_id'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- op6 -- Spalten definieren
      WHEN pg_typeof(null::op6) THEN
        cols_check :=
          (
            '{'
              'o6_aknr, o6_o2_n, o6_pos, o6_ap_nr, '
              'o6_zaknr, o6_calconly, '
              'o6_txt, o6_txt_rtf, o6_txtbem, o6_txtbem_rtf, '
              'o6_folgebearb1, o6_folgebearb1_rtf, o6_folgebearb2, o6_folgebearb2_rtf, o6_folgebearb3, o6_folgebearb3_rtf, '
              'o6_m, o6_m_stat, o6_stat, '
              'o6_ekenner_krz, o6_m_fb_name, o6_m_fertdata, '
              'o6_mce, o6_stkz, o6_lz, o6_bz, o6_hz, o6_zz, o6_zme, o6_min, o6_dimi'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- stv -- Spalten definieren
      WHEN pg_typeof(null::stv) THEN
        cols_check :=
          (
            '{'
              'st_zn, st_n, st_pos, st_pos_parent, '
              'st_mgc, st_m, st_m_uf1, st_m_fix, '
              'st_txt, st_stat, st_ekenner_krz, st_kstv_st_id, st_posinfo'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- epreis -- Spalten definieren
      WHEN pg_typeof(null::epreis) THEN
        cols_check :=
          (
            '{'
              'e_aknr, e_fertaknr, e_lkn, e_best ,e_herkunft, '
              'e_txt, e_txt_rtf, '
              'e_folgeap_op_ix, e_folgeap_op_ix__disabled'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- ab2 -- Spalten definieren
      WHEN pg_typeof(null::ab2) THEN
        cols_check :=
          (
            '{'
              'a2_n, a2_ks, a2_txt, a2_txt_rtf'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- ab2 -- Spalten definieren
      WHEN pg_typeof(null::ab2_wkstplan) THEN
        cols_check :=
          (
            '{'
              'a2w_oks'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- auftg -- Spalten definieren
      WHEN pg_typeof(null::auftg) THEN
        cols_check :=
          (
            '{'
              'ag_aknr,  ag_aknr_idx,  ag_akbz,  ag_aknr_referenz,  '
              'ag_stk,  '
              'ag_mcv,  '
              'ag_o6_dimi,  ag_o6_stkz,  ag_o6_lz,  ag_o6_bz,  ag_o6_hz,  ag_o6_zz,  ag_o6_zme,  ag_o6_pos'
            '}'
          )::VARCHAR[]
        ;
        cols_ignore := '{}'::VARCHAR[];

      -- zB Recnocomment per Default alle Spalten (außer System in ZFunc bereits abgefangen)
      ELSE
        -- this will cause all columns to be checked, except for the system columns (dbrid, modified, etcpp)
        cols_check  := '{}'::VARCHAR[];
        cols_ignore := '{}'::VARCHAR[];
    END CASE;
  END $$ LANGUAGE plpgsql IMMUTABLE;


--
--------------------------------------------------------------------------------
-- the trigger function to be attached to every table for which secpol_check needs to be performed for insert, update, delete
-- this is for non-customer-specific chain !!!
-- keep this function equivalent to TSystem.alltables__b_iud__secpol_customer_check() (with difference: will call secpol_customer_check)
CREATE OR REPLACE FUNCTION TSystem.alltables__b_iud__secpol_check()
  RETURNS TRIGGER
  SET search_path = public, TSystem
  AS $$
  DECLARE
    _rec_old     JSONB;
    _rec_current JSONB;

    _cols_ignore VARCHAR[];
    _cols_check  VARCHAR[];

    _ok          BOOL DEFAULT true;
    _ignore      BOOL DEFAULT false;
    _wrong_op    BOOL DEFAULT false;
    _err_hint    VARCHAR;

  BEGIN
    CASE tg_op
      WHEN 'INSERT' THEN
        _rec_old     := '{}'::JSONB;
        _rec_current := to_jsonb(new);

      WHEN 'UPDATE' THEN
        EXECUTE 'SELECT (TSystem.secpol_checkcolumns_get($1)).*' INTO _cols_check, _cols_ignore USING old;

        _rec_old     := to_jsonb(old);
        _rec_current := to_jsonb(new);

        _ignore := TSystem.secpol_compare_records_oldnew(_rec_old, _rec_current, _cols_check, _cols_ignore);

      WHEN 'DELETE' THEN
        _rec_old     := to_jsonb(old);
        _rec_current := '{}'::JSONB;

      ELSE
        _wrong_op := true;
        PERFORM TSystem.secpol_raise_exception(_wrong_op, NULL, 'wrong op in trigger'::TEXT, NULL, 'C0001'::TEXT, NULL, NULL, tg_relname, NULL);
    END CASE;

    IF (NOT(_ignore)) THEN
      EXECUTE
        'SELECT (secpol_check($1, $2, $3)).*'
      INTO
        _ok,
        _err_hint
      USING
        coalesce(new, old), _rec_old, _rec_current
      ;
      _ok := coalesce(_ok, false);

      PERFORM TSystem.secpol_raise_exception(NOT(_ok), NULL, NULL, _err_hint::TEXT, 'C0001'::TEXT, NULL, NULL, tg_relname, NULL);
    END IF;

    -- default handling when no error
    IF tg_op = 'DELETE' THEN
      RETURN old;
    ELSE -- INSERT, UPDATE
      RETURN new;
    END IF;
  END $$ LANGUAGE plpgsql;


--
--------------------------------------------------------------------------------
-- the trigger function to be attached to every table for which secpol_customer_check needs to be performed for insert, update, delete
-- this is for customer-specific chain !!!
-- keep this function equivalent to TSystem.alltables__b_iud__secpol_check() (with difference: will call secpol_check)
CREATE OR REPLACE FUNCTION TSystem.alltables__b_iud__secpol_customer_check()
  RETURNS TRIGGER
  SET search_path = public, TSystem, z_50_customer
  AS $$
  DECLARE
    _rec_old     JSONB;
    _rec_current JSONB;

    _cols_ignore VARCHAR[];
    _cols_check  VARCHAR[];

    _ok          BOOL DEFAULT true;
    _ignore      BOOL DEFAULT false;
    _wrong_op    BOOL DEFAULT false;
    _err_hint    VARCHAR;

  BEGIN
    CASE tg_op
      WHEN 'INSERT' THEN
        _rec_old     := '{}'::JSONB;
        _rec_current := to_jsonb(new);

      WHEN 'UPDATE' THEN
        EXECUTE 'SELECT (TSystem.secpol_checkcolumns_get($1)).*' INTO _cols_check, _cols_ignore USING old;

        _rec_old     := to_jsonb(old);
        _rec_current := to_jsonb(new);

        _ignore := TSystem.secpol_compare_records_oldnew(_rec_old, _rec_current, _cols_check, _cols_ignore);

      WHEN 'DELETE' THEN
        _rec_old     := to_jsonb(old);
        _rec_current := '{}'::JSONB;

      ELSE
        _wrong_op := true;
        PERFORM TSystem.secpol_raise_exception(_wrong_op, NULL, 'wrong op in trigger'::TEXT, NULL, 'C0001'::TEXT, NULL, NULL, tg_relname, NULL);
    END CASE;

    IF (NOT(_ignore)) THEN
      EXECUTE
        'SELECT (secpol_customer_check($1, $2, $3)).*'
      INTO
        _ok,
        _err_hint
      USING
        coalesce(new, old), _rec_old, _rec_current
      ;
      _ok := coalesce(_ok, false);

      PERFORM TSystem.secpol_raise_exception(NOT(_ok), NULL, NULL, _err_hint::TEXT, 'C0001'::TEXT, NULL, NULL, tg_relname, NULL);
    END IF;

    -- default handling when no error
    IF tg_op = 'DELETE' THEN
      RETURN old;
    ELSE -- INSERT, UPDATE
      RETURN new;
    END IF;
  END $$ LANGUAGE plpgsql;